/*

Pinky_waveform.js:

opens a MsPinky-style waveform file, reads the data, and displays in a jsgui
Very basic.  Uses the default "sketch" instance to display the waveform, while using
another instance (AuxSketch) to draw the waveform in chunks (since the size of a
Sketch object is annoyingly limited to 2048).  The completed waveform drawing is stored
in an Image object (AuxImage) so that it needn't be re-computed every time the user
scrolls.

Author: Scott Wardle aka da misteryus P. Minsky
Date: August 24, 2004

THIS is the 060331 mac version slightly modified by sagawee to work on windows

*/

outlets = 5;
sketch.default2d();
sketch.fsaa = false;
setoutletassist(4,"to MPTCFS~ left inlet");
setoutletassist(3,"waveform max");
setoutletassist(2,"waveform min");
setoutletassist(1,"file size (in bytes)");
setoutletassist(0,"file name");

var theWaveformFile = 0;
var theWaveformTask = 0;
var is_waveform_calculated = false;
var busy_calculating_waveform = false;
var percent_waveform = 0.0;
var waveform_data_found = false;
var autocalculate_waveform = false;
var theWaveformIndexBuff = new Array(20);
var WaveformMin = -1.0;
var WaveformMax = -1.0;

var WaveformZoom = 1;
var MaxWaveformZoom = 16;
var MinWaveformZoom = 1;
var WaveformOffset = 0;
var WaveformAutoScroll = false;
var WaveformContinuousScroll = true;
var PositionIndicator = 0;
var AuxPositionIndicator = -1;

var AuxSketch = 0;
var AuxImage = 0;
var IndicatorImage = 0;
var AuxIndicatorImage = 0;
var Drag_Happened = false;

var background_color = new Array(0.0,0.0,0.0);
var waveform_color = new Array(0.0,0.75,0.0);
var indicator_color = new Array(1.0,1.0,1.0);
var aux_color = new Array(1.0,0.0,0.0);

function setzoom(v)
{
	if ((v >= MinWaveformZoom) && (v <= MaxWaveformZoom)) {
		WaveformZoom = v;
		var width = box.rect[2] - box.rect[0]; 
		var height = box.rect[3] - box.rect[1]; 
							
		if (AuxSketch) delete AuxSketch;
		AuxSketch = new Sketch(Math.min(width*WaveformZoom,1024),height);
		AuxSketch.fsaa = false;
		
		AuxSketch.default2d();
		AuxSketch.glclearcolor(background_color[0],background_color[1],background_color[2]);
		AuxSketch.glclear();			
		AuxSketch.glcolor(waveform_color[0],waveform_color[1],waveform_color[2],1);
		
		if (AuxImage) delete AuxImage;
		AuxImage = new Image(width*WaveformZoom,height);
		
		if (IndicatorImage) delete IndicatorImage;
		IndicatorImage = new Image(1,height);

		if (AuxIndicatorImage) delete AuxIndicatorImage;
		AuxIndicatorImage = new Image(1,height);
		
		// set the indicator image to all white pixels
		for (i=0; i<height; i++) {
			IndicatorImage.setpixel(0,i,indicator_color[0],indicator_color[1],indicator_color[2],1.0);
			AuxIndicatorImage.setpixel(0,i,aux_color[0],aux_color[1],aux_color[2],1.0);
		}
		
		draw_aux();
		draw();
		refresh();
	}
}

function autoscroll(v)
{
	if (v == 0) {
		WaveformAutoScroll = false;
	} else if (v > 0) {
		WaveformAutoScroll = true;
	}
}

function contscroll(v)
{
	if (v == 0) {
		WaveformContinuousScroll = false;
	} else if (v > 0) {
		WaveformContinuousScroll = true;
	}
}

function autocalculate(v)
{
	if (v == 0) {
		autocalculate_waveform = false;
	} else if (v > 0) {
		autocalculate_waveform = true;
	}
}

function onresize(width,height)
{
	setzoom(WaveformZoom);
} 
onresize.local = 1; //private

function set_background_color(r,g,b)
{
	background_color[0] = r/255.0;
	background_color[1] = g/255.0;
	background_color[2] = b/255.0;
	setzoom(WaveformZoom);
}

function set_waveform_color(r,g,b)
{
	waveform_color[0] = r/255.0;
	waveform_color[1] = g/255.0;
	waveform_color[2] = b/255.0;
	setzoom(WaveformZoom);
}

function scroll_waveform(v)
{
	var do_indicator = false;
	
	if (theWaveformTask) {
		if (!theWaveformTask.running) {
			do_indicator = true;
		}
	} else do_indicator = true;
	
	if (do_indicator) {
		erase_indicator();
		if (v < 0.0) {
			WaveformOffset = 0;
		} else if (v > 1.0) {
			WaveformOffset = AuxImage.size[0] - sketch.size[0];
		} else {
			WaveformOffset = Math.round(v*(AuxImage.size[0] - sketch.size[0]));
		}
		draw();
		refresh();
		outlet(4,"waveformoffset",WaveformOffset/(AuxImage.size[0]-sketch.size[0]));
	}
}

function update_indicator(v)
{
	var do_indicator = false;
	
	if (theWaveformTask) {
		if (!theWaveformTask.running) {
			do_indicator = true;
		}
	} else do_indicator = true;
	
	if (do_indicator) {
		erase_indicator();
		if (v < 0.0) {
			PositionIndicator = 0;
		} else if (v > 1.0) {
			PositionIndicator = AuxImage.size[0];
		} else {
			PositionIndicator = Math.round(v*AuxImage.size[0]);
		}
		
		if (WaveformContinuousScroll) {
			WaveformOffset = PositionIndicator - (0.5*sketch.size[0]);
			WaveformOffset = Math.min(WaveformOffset,AuxImage.size[0]-sketch.size[0]);
			WaveformOffset = Math.max(WaveformOffset,0);	
			draw();		
		} else {
			draw_indicator();
		}
		
		refresh();
		outlet(4,"waveformoffset",WaveformOffset/(AuxImage.size[0]-sketch.size[0]));
	}
}

function clear_display()
{
	if (theWaveformTask) {
		if (theWaveformTask.running) theWaveformTask.cancel();
	}
	
	if (theWaveformFile) {
		theWaveformFile.close();
		delete theWaveformfile;
		theWaveformFile = 0;
	}
	
	draw();
	refresh();
}

function open_waveform_file(v)
{
	if (theWaveformFile) {
		theWaveformFile.close();
		delete theWaveformfile;
	} 
	theWaveformFile = new File(v,"read");
	waveform_data_found = false;
	if ((theWaveformFile.isopen) &&
		(theWaveformFile.eof > 120)) {
		theWaveformFile.position = 0;
		var string1 = theWaveformFile.readstring(4);
		var string2 = theWaveformFile.readstring(4);
		var string3 = theWaveformFile.readstring(4);
		var string4 = theWaveformFile.readstring(4);						
		if ((string1 == "BABY") &&
			(string2 == "TALK") &&
			(string3 == "WAVE") &&
			(string4 == "FORM")) {
			WaveformMin = theWaveformFile.readfloat32(1);
			WaveformMax = theWaveformFile.readfloat32(1);
			
			/* Skip over the next four longs */
			theWaveformFile.readint32(4);
			
			/* the next 20 longs should contain the waveform index table*/
			theWaveformIndexBuff = theWaveformFile.readint32(20);

			var width = box.rect[2] - box.rect[0]; 
			
	  		// We must check to see if the number of lines provided in the file
	  		// at various resolutions will provide what we need to create our zoomed views
	  		// Start by looking for a data segment with at least the max number of
	  		// lines we will need to draw, but less than 2X the max number of lines.
	  		// Then look for the minimum number of lines we'll need.
	  		var byte_offset = 120;
	  		var max_num_lines = width*MaxWaveformZoom;  // or whatever maximum zoom factor currently is
	  		
	  		max_num_lines = Math.min(max_num_lines, 16384);
	  		
	  		var temp_index = 0;
	  		var loc_found_max_count = false;
	  		var loc_found_counts = false;
	  		while ((loc_found_max_count == false) && 
	  				(temp_index < 20) && 
	  				(theWaveformIndexBuff[temp_index] == byte_offset)) {
	  			if ((theWaveformIndexBuff[temp_index+1] >= max_num_lines) &&
	  				(theWaveformIndexBuff[temp_index+1] <= 2*max_num_lines)) {
	  				loc_found_max_count = true;
	  				break;	
	  			}		
	  			byte_offset += theWaveformIndexBuff[temp_index+1];
	  			temp_index += 2;
	  		}	

			if (loc_found_max_count) {		  		
				var min_num_lines = width;
				
				while ((loc_found_counts == false) &&
					   (temp_index < 20) &&
					   (theWaveformIndexBuff[temp_index] == byte_offset)) {  
					byte_offset += theWaveformIndexBuff[temp_index+1];
					temp_index += 2;
				}
				
				if (temp_index == 20) {
					waveform_data_found = true;		
					draw_aux();
					draw();
					refresh();
					outlet(3,WaveformMax);
					outlet(2,WaveformMin);
					outlet(1,theWaveformFile.eof);
					outlet(0,theWaveformFile.filename);			
				} else {
					post("Pinky_waveform: failed to find all counts in " + v + "\n");
					draw();
					refresh();
				}
			} else {
				post("Pinky_waveform: failed to find max count in " + v + "\n");
				draw();
				refresh();				
			}
		}
	} else {
		draw();
		refresh();
	}
	
	if (autocalculate_waveform && !waveform_data_found) {
		start_waveform_calc();
	}
}

function draw_aux()
{
	if (waveform_data_found) {
		var temp_index = -1;
		// Find the section of waveform data of the appropriate width for our display
		while ((theWaveformIndexBuff[temp_index+2] >= 2*AuxImage.size[0]) &&
			(temp_index < 19)) {
			temp_index += 2;
		}
		if (temp_index < 0) temp_index = 1;

		var theWaveformData;
		var temp_array = new Array(2048);
		
	//	post("temp_index = " + temp_index + "\n");
	//	post("indexBuff[temp_index] = " + theWaveformIndexBuff[temp_index] + "\n");
	//	post("indexBuff[temp_index-1] = " + theWaveformIndexBuff[temp_index-1] + "\n");		
	//	post("file position = " + theWaveformFile.position + "\n");
		theWaveformFile.position = theWaveformIndexBuff[temp_index-1];
	//	post("file position(after) = " + theWaveformFile.position + "\n");		
		
		var num_read = 0;
		var num2read = Math.min(2048, theWaveformIndexBuff[temp_index]);
		theWaveformData = theWaveformFile.readbytes(num2read);
		num_read += theWaveformData.length;
		while (num_read < theWaveformIndexBuff[temp_index]) {
			num2read = Math.min(2048, theWaveformIndexBuff[temp_index] - num_read);
			temp_array = theWaveformFile.readbytes(num2read);
			theWaveformData = theWaveformData.concat(temp_array);
			num_read += temp_array.length;
		}
	//	post("theWaveformData.length = " + theWaveformData.length + "\n");
		var num_lines = Math.floor(theWaveformData.length/2);
	//	post("num_lines = " + num_lines + "\n");
		
		// Draw into the AuxSketch
		AuxSketch.glclearcolor(background_color[0],background_color[1],background_color[2]);
		AuxSketch.glclear();			
		AuxSketch.glcolor(waveform_color[0],waveform_color[1],waveform_color[2],1);

		var image_aspect = AuxImage.size[0]/AuxImage.size[1];
	//	post("image aspect = " + image_aspect + "\n");
		
		var sketch_aspect = AuxSketch.size[0]/AuxSketch.size[1];
	//	post("sketch aspect = " + sketch_aspect + "\n");
						
		var x_increment = (2.0*image_aspect)/num_lines;
		var xlo = -sketch_aspect;
		var ylo = 0.0;
		var yhi = 0.0;
		var scaler = 1.0/128.0;
		var destination_x = 0;
		
	//	post("x_increment = " + x_increment + "\n");
	//	post("xlo" + xlo + "\n");
	//	post("scaler" + scaler + "\n");
		for (i=0; i<num_lines; i++) {
			ylo = -1.0+scaler*theWaveformData[i*2];
			yhi = -1.0+scaler*theWaveformData[(i*2)+1];
			AuxSketch.linesegment(xlo,ylo,0,xlo,yhi,0);		
			xlo += x_increment;		
			if (xlo > sketch_aspect) {
//				post("copying stuff from sketch to image: xlo=" + xlo + ", dest_x=" + destination_x + ", i=" + i + "\n");
				// the sketch object is full.. copy to image
				AuxImage.copypixels(AuxSketch,destination_x,0,0,0,AuxSketch.size[0],AuxSketch.size[1]);
				xlo = -sketch_aspect;
				destination_x += AuxSketch.size[0];
				AuxSketch.glclear();			
			}
		} 
		
		// check to see if there is any left over stuff in the sketch object
		if (xlo > -sketch_aspect) {			
			var leftover_x = Math.min(AuxImage.size[0] - destination_x, AuxSketch.size[0]);
		
//			post("copying " + leftover_x + " leftover lines from sketch, \n");	
//			post("destination_x=" + destination_x + "\n");		
			AuxImage.copypixels(AuxSketch,destination_x,0,0,0,leftover_x,AuxSketch.size[1]);
		}
	}	
}
draw_aux.local = 1; //private

function draw()
{
	if (waveform_data_found) {
		//Just copy the pixels from the image object to the default instance of sketch
		sketch.copypixels(AuxImage,0,0,WaveformOffset,0,sketch.size[0],sketch.size[1]);
		draw_indicator();	
	} else {
		var do_background = false;
		
		if (theWaveformTask) {
			if (!theWaveformTask.running) {
				do_background = true;	
			}
		} else {
			do_background = true;
		}
		
		if (do_background) {
			// Just fill in a black background with a flat line across
			sketch.glclearcolor(background_color[0],background_color[1],background_color[2]);
			sketch.glclear();			
		
			sketch.glcolor(waveform_color[0],waveform_color[1],waveform_color[2],1);
			sketch.gllinewidth(1);
		
			var aspect = sketch.size[0]/sketch.size[1];
			sketch.linesegment(-aspect,0,0,aspect,0,0);	
			draw_indicator();	
		}
	}	
}
draw.local = 1; //private

function erase_indicator()
{
	// copy pixels over the old position indicator
	sketch.copypixels(AuxImage,PositionIndicator - WaveformOffset,0,PositionIndicator,0,1,sketch.size[1]);
}
erase_indicator.local = 1; //private

function erase_aux_indicator()
{
 	if ((AuxPositionIndicator >= WaveformOffset) && (AuxPositionIndicator <= WaveformOffset + sketch.size[0])) {
 		// erase the old aux indicator position by copying pixels back over it
 		sketch.copypixels(AuxImage,AuxPositionIndicator - WaveformOffset,0,AuxPositionIndicator,0,1,sketch.size[1]);
 	}
}
erase_aux_indicator.local = 1; //private

function draw_indicator()
{
	if ((PositionIndicator >= WaveformOffset) && (PositionIndicator <= WaveformOffset + sketch.size[0])) {
		// the new position is within the currently viewed portion of the waveform
		// we "draw" the indicator by copying pixels from the indicator image;
		sketch.copypixels(IndicatorImage,PositionIndicator - WaveformOffset,0,0,0,1,sketch.size[1]);
	} else {
		if (WaveformAutoScroll == true) {
			var distance = PositionIndicator - WaveformOffset;
			
			if ((distance > 0) && (distance <= 2*sketch.size[0])) {
				WaveformOffset += sketch.size[0];
				WaveformOffset = Math.min(WaveformOffset,AuxImage.size[0]-sketch.size[0]);
			} else if ((distance < 0) && (distance > -sketch.size[0])) {
				WaveformOffset -= sketch.size[0];
				WaveformOffset = Math.max(WaveformOffset,0);
			} else {
				WaveformOffset = PositionIndicator - (0.5*sketch.size[0]);
				WaveformOffset = Math.min(WaveformOffset,AuxImage.size[0]-sketch.size[0]);
				WaveformOffset = Math.max(WaveformOffset,0);				
			}
			draw();
			refresh();
		}
	}
}
draw_indicator.local = 1; // private function

function waveform_calculated(v)
{
	if (v) {
		is_waveform_calculated = true;	
	} else {
		is_waveform_calculated = false;
	}
}

function percent_waveform_calculated(v)
{
	percent_waveform = v;
	if (busy_calculating_waveform) theWaveformTask.schedule(20);	
}

function waveform_repeater()
{
	if (is_waveform_calculated) {
		arguments.callee.task.cancel();
		busy_calculating_waveform = false;
		outlet(4,"query","waveform_filename");
	} else {		
		var aspect = sketch.size[0]/sketch.size[1];
		var limit = aspect*(-1.0 + 2.0*percent_waveform);
			
		sketch.glcolor(waveform_color[0],waveform_color[1],waveform_color[2],1);
		sketch.quad(-aspect,-1,0,-aspect,1,0,limit,1,0,limit,-1,0);			
		sketch.fontsize(12);
		sketch.glcolor(1.0,1.0,1.0);
		sketch.moveto(-0.9*aspect,0.0,0.0);		
		sketch.text("calculating waveform data...");			
		refresh();
		busy_calculating_waveform = true;
		outlet(4,"waveformcalc",2);
	}
}
waveform_repeater.local = 1; // private

function start_waveform_calc()
{	
	if (theWaveformFile) {	
			// begin the task of computing the waveform
			if (!theWaveformTask) theWaveformTask = new Task(waveform_repeater);
			is_waveform_calculated = false;
			percent_waveform = 0.0;
			sketch.glclearcolor(background_color[0],background_color[1],background_color[2]);
			sketch.glclear();			
			refresh();
			busy_calculating_waveform = true;
			theWaveformTask.schedule(100);
	}
} 
start_waveform_calc.local = 1; //private

function onclick(x,y,button,mod1,shift,caps,opt,mod2)
{
	if (shift && !waveform_data_found && theWaveformFile.isopen) {
		start_waveform_calc();
	}
}
onclick.local = 1; // private function

function ondblclick(x,y,button,mod1,shift,caps,opt,mod2)
{
	if (shift && !waveform_data_found) {
		start_waveform_calc();
	} else {
		erase_indicator();
		PositionIndicator = WaveformOffset + x;
		draw_indicator();
		refresh();	
		outlet(4,"cue",PositionIndicator/AuxImage.size[0]);
		outlet(4,"query","msec_time");
		outlet(4,"query","file_time");
	}
}
ondblclick.local = 1; // private function

function ondrag(x,y,button,cmd,shift,capslock,option,ctrl)
{
 	if (button) {
 		Drag_Happened = true;
	 	erase_aux_indicator();
 	
		AuxPositionIndicator = WaveformOffset + x;
 		
 		// redraw the regular indicator
  		draw_indicator();		
 		
 		// draw the new aux position indicator by copying pixels
 		sketch.copypixels(AuxIndicatorImage,AuxPositionIndicator - WaveformOffset,0,0,0,1,sketch.size[1]);
 	} else {
 		if (Drag_Happened) {
	 		// the aux indicator now becomes the indicator
			erase_indicator();
		
			PositionIndicator =  AuxPositionIndicator;
		
			draw_indicator();
 			AuxPositionIndicator = -1;
			outlet(4,"cue",PositionIndicator/AuxImage.size[0]);
			outlet(4,"query","msec_time");
			outlet(4,"query","file_time");	
 			Drag_Happened = false;
 		}
 	}
 	refresh();	
}
ondrag.local = 1; //private. could be left public to permit "synthetic" events

function loadbang()
{
	setzoom(1);
}
